<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class WPSC_Download_Tracker {

    public function __construct() {
        // Download-Links im Content ersetzen (alle Dateitypen + externe URLs)
        add_filter( 'the_content', array( $this, 'replace_download_links' ), 20 );

        // AJAX: Download zählen
        add_action( 'wp_ajax_wpsc_track_download',        array( $this, 'ajax_track_download' ) );
        add_action( 'wp_ajax_nopriv_wpsc_track_download', array( $this, 'ajax_track_download' ) );

        // AJAX: Download-Count nur lesen (kein Zählen)
        add_action( 'wp_ajax_wpsc_get_download_count',        array( $this, 'ajax_get_download_count' ) );
        add_action( 'wp_ajax_nopriv_wpsc_get_download_count', array( $this, 'ajax_get_download_count' ) );
    }

    // ── Links im Content ersetzen ─────────────────────────────────────
    public function replace_download_links( $content ) {
        if ( ! is_singular() ) return $content;

        // Findet alle <a href="..."> Tags
        $pattern = '/<a\s([^>]*href=["\']([^"\']+)["\'][^>]*)>/i';
        $content = preg_replace_callback( $pattern, array( $this, 'process_link' ), $content );
        return $content;
    }

    private function process_link( $matches ) {
        $full_tag  = $matches[0];
        $attrs     = $matches[1];
        $url       = $matches[2];

        // Interne WordPress-Navigation ausschließen (nur #anchor, ?page_id= etc.)
        if ( $this->is_navigation_link( $url ) ) {
            return $full_tag;
        }

        // Prüfen ob es ein Download-Link ist
        if ( ! $this->is_download_link( $url ) ) {
            return $full_tag;
        }

        // data-Attribute hinzufügen für JS-Tracking
        $file_name = basename( parse_url( $url, PHP_URL_PATH ) );
        $post_id   = get_the_ID();

        // Bestehende data-wpsc-download nicht doppelt setzen
        if ( strpos( $attrs, 'data-wpsc-download' ) !== false ) {
            return $full_tag;
        }

        $new_attrs = $attrs . sprintf(
            ' data-wpsc-download="1" data-wpsc-url="%s" data-wpsc-post="%d" data-wpsc-file="%s"',
            esc_attr( $url ),
            absint( $post_id ),
            esc_attr( $file_name )
        );

        return '<a ' . $new_attrs . '>';
    }

    // ── Erkennung: Ist der Link ein Download-Link? ────────────────────
    private function is_download_link( $url ) {
        // Alle nicht-HTML Dateien werden als Download gewertet
        // Typische Web-Seiten-Endungen ausschließen
        $excluded_extensions = array( 'html', 'htm', 'php', 'asp', 'aspx' );

        $path = parse_url( $url, PHP_URL_PATH );
        if ( empty( $path ) ) return false;

        $ext = strtolower( pathinfo( $path, PATHINFO_EXTENSION ) );

        // Wenn keine Endung → könnte externe URL oder Seite sein
        // Externe URLs mit Dateiendung tracken
        if ( empty( $ext ) ) {
            // Externe URLs ohne Endung NICHT tracken (normale Seiten)
            $site_url = parse_url( home_url(), PHP_URL_HOST );
            $link_host = parse_url( $url, PHP_URL_HOST );
            if ( $link_host && $link_host !== $site_url ) {
                // Externe URL ohne Dateiendung tracken
                return true;
            }
            return false;
        }

        // Ausgeschlossene Web-Endungen
        if ( in_array( $ext, $excluded_extensions ) ) {
            return false;
        }

        return true;
    }

    // ── Navigations-Links ausschließen ────────────────────────────────
    private function is_navigation_link( $url ) {
        // Nur Anker (#section)
        if ( strpos( $url, '#' ) === 0 ) return true;
        // JavaScript-Links
        if ( strpos( $url, 'javascript:' ) === 0 ) return true;
        // mailto / tel
        if ( strpos( $url, 'mailto:' ) === 0 ) return true;
        if ( strpos( $url, 'tel:' ) === 0 ) return true;
        return false;
    }

    // ── AJAX: Download-Zähler erhöhen ─────────────────────────────────
    public function ajax_track_download() {
        check_ajax_referer( 'wpsc_nonce', 'nonce' );

        $url       = isset( $_POST['url'] )       ? esc_url_raw( $_POST['url'] ) : '';
        $post_id   = isset( $_POST['post_id'] )   ? absint( $_POST['post_id'] )  : 0;
        $file_name = isset( $_POST['file_name'] ) ? sanitize_text_field( $_POST['file_name'] ) : '';

        if ( empty( $url ) ) {
            wp_send_json_error( 'No URL' );
        }

        // Cookie-Schutz
        $cookie_key  = 'wpsc_dl_' . md5( $url );
        if ( isset( $_COOKIE[ $cookie_key ] ) ) {
            $count = self::get_download_count( $url );
            wp_send_json_success( array( 'count' => $count, 'cached' => true ) );
        }

        // Zähler erhöhen
        $count = self::increment_download( $post_id, $url, $file_name );

        // Cookie
        $expires = time() + ( absint( get_option( 'wpsc_cookie_days', 1 ) ) * DAY_IN_SECONDS );
        setcookie( $cookie_key, '1', $expires, COOKIEPATH, COOKIE_DOMAIN, is_ssl(), true );

        wp_send_json_success( array( 'count' => $count ) );
    }

    // ── AJAX: Nur lesen, nicht zählen ─────────────────────────────────
    public function ajax_get_download_count() {
        check_ajax_referer( 'wpsc_nonce', 'nonce' );
        $url = isset( $_POST['url'] ) ? esc_url_raw( $_POST['url'] ) : '';
        if ( empty( $url ) ) {
            wp_send_json_error( 'No URL' );
        }
        $count = self::get_download_count( $url );
        wp_send_json_success( array( 'count' => $count ) );
    }

    // ── Download-Count für eine URL ───────────────────────────────────
    public static function get_download_count( $url ) {
        global $wpdb;
        $table = $wpdb->prefix . 'wpsc_downloads';
        $count = $wpdb->get_var(
            $wpdb->prepare( "SELECT download_count FROM {$table} WHERE file_url = %s LIMIT 1", $url )
        );
        return $count ? absint( $count ) : 0;
    }

    // ── Zähler erhöhen ────────────────────────────────────────────────
    public static function increment_download( $post_id, $url, $file_name = '' ) {
        global $wpdb;
        $table = $wpdb->prefix . 'wpsc_downloads';

        $existing = $wpdb->get_row(
            $wpdb->prepare( "SELECT id, download_count FROM {$table} WHERE file_url = %s LIMIT 1", $url )
        );

        if ( $existing ) {
            $wpdb->update(
                $table,
                array( 'download_count' => $existing->download_count + 1, 'last_download' => current_time( 'mysql' ) ),
                array( 'id' => $existing->id ),
                array( '%d', '%s' ),
                array( '%d' )
            );
            return $existing->download_count + 1;
        } else {
            $wpdb->insert(
                $table,
                array(
                    'post_id'        => $post_id,
                    'file_url'       => $url,
                    'file_name'      => $file_name ?: basename( parse_url( $url, PHP_URL_PATH ) ),
                    'download_count' => 1,
                    'last_download'  => current_time( 'mysql' ),
                ),
                array( '%d', '%s', '%s', '%d', '%s' )
            );
            return 1;
        }
    }

    // ── Top-Downloads für Dashboard ───────────────────────────────────
    public static function get_top_downloads( $limit = 20 ) {
        global $wpdb;
        $table = $wpdb->prefix . 'wpsc_downloads';
        return $wpdb->get_results(
            $wpdb->prepare(
                "SELECT d.*, p.post_title
                 FROM {$table} d
                 LEFT JOIN {$wpdb->posts} p ON p.ID = d.post_id
                 ORDER BY d.download_count DESC
                 LIMIT %d",
                $limit
            )
        );
    }

    // ── Gesamt-Downloads ──────────────────────────────────────────────
    public static function get_total_downloads() {
        global $wpdb;
        $table = $wpdb->prefix . 'wpsc_downloads';
        return (int) $wpdb->get_var( "SELECT SUM(download_count) FROM {$table}" );
    }
}
