<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class AutoSEO_Processor {

    private $generator;

    public function __construct() {
        $this->generator = new AutoSEO_Generator();
    }

    public function init() {
        if ( get_option( 'autoseo_auto_on_save', '1' ) === '1' ) {
            add_action( 'save_post', array( $this, 'on_save_post' ), 20, 2 );
        }
        add_action( 'wp_ajax_autoseo_process_single', array( $this, 'ajax_single' ) );
        add_action( 'wp_ajax_autoseo_process_all',    array( $this, 'ajax_all' ) );
        add_action( 'wp_ajax_autoseo_get_missing',    array( $this, 'ajax_missing' ) );
        add_action( 'wp_ajax_autoseo_clear_log',      array( $this, 'ajax_clear_log' ) );
    }

    public function on_save_post( $post_id, $post ) {
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) return;
        if ( wp_is_post_revision( $post_id ) ) return;
        if ( $post->post_status !== 'publish' ) return;
        $allowed = get_option( 'autoseo_post_types', array( 'post', 'page' ) );
        if ( ! in_array( $post->post_type, $allowed ) ) return;

        $overwrite = get_option( 'autoseo_overwrite', '0' ) === '1';
        $has_all   = $this->has_all_fields( $post_id );
        if ( $has_all && ! $overwrite ) return;

        remove_action( 'save_post', array( $this, 'on_save_post' ), 20 );
        $this->process( $post_id, $post->post_title, $post->post_content, $post->post_type, 'auto_save' );
        add_action( 'save_post', array( $this, 'on_save_post' ), 20, 2 );
    }

    public function ajax_single() {
        check_ajax_referer( 'autoseo_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) wp_die();
        $post_id = intval( $_POST['post_id'] ?? 0 );
        $post    = $post_id ? get_post( $post_id ) : null;
        if ( ! $post ) wp_send_json_error( 'Beitrag nicht gefunden.' );
        $result = $this->process( $post_id, $post->post_title, $post->post_content, $post->post_type, 'manual' );
        is_wp_error( $result )
            ? wp_send_json_error( $result->get_error_message() )
            : wp_send_json_success( array( 'post_id' => $post_id, 'title' => $post->post_title, 'data' => $result ) );
    }

    public function ajax_all() {
        check_ajax_referer( 'autoseo_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) wp_die();

        $offset    = intval( $_POST['offset'] ?? 0 );
        $overwrite = get_option( 'autoseo_overwrite', '0' ) === '1';
        $allowed   = get_option( 'autoseo_post_types', array( 'post', 'page' ) );

        $posts = get_posts( array(
            'post_type' => $allowed, 'post_status' => 'publish',
            'posts_per_page' => 10, 'offset' => $offset,
            'orderby' => 'date', 'order' => 'DESC',
        ) );

        if ( empty( $posts ) ) {
            wp_send_json_success( array( 'done' => true, 'processed' => 0 ) );
        }

        $processed = array(); $skipped = 0;
        foreach ( $posts as $post ) {
            if ( $this->has_all_fields( $post->ID ) && ! $overwrite ) { $skipped++; continue; }
            $r = $this->process( $post->ID, $post->post_title, $post->post_content, $post->post_type, 'bulk' );
            if ( ! is_wp_error( $r ) ) {
                $processed[] = array( 'post_id' => $post->ID, 'title' => $post->post_title, 'data' => $r );
            }
        }

        wp_send_json_success( array(
            'done' => false, 'processed' => count( $processed ),
            'skipped' => $skipped, 'results' => $processed,
            'next_offset' => $offset + 10,
        ) );
    }

    public function ajax_missing() {
        check_ajax_referer( 'autoseo_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) wp_die();

        $allowed   = get_option( 'autoseo_post_types', array( 'post', 'page' ) );
        $all_posts = get_posts( array(
            'post_type' => $allowed, 'post_status' => 'publish',
            'posts_per_page' => -1, 'orderby' => 'date', 'order' => 'DESC',
        ) );

        $missing = array(); $complete = 0;
        foreach ( $all_posts as $post ) {
            $m = $this->missing_fields( $post->ID );
            if ( ! empty( $m ) ) {
                $missing[] = array(
                    'id' => $post->ID, 'title' => $post->post_title,
                    'type' => $post->post_type, 'missing_fields' => $m,
                    'edit_url' => get_edit_post_link( $post->ID ),
                );
            } else { $complete++; }
        }

        wp_send_json_success( array(
            'missing' => $missing, 'missing_count' => count( $missing ),
            'complete_count' => $complete, 'total' => count( $all_posts ),
        ) );
    }

    public function ajax_clear_log() {
        check_ajax_referer( 'autoseo_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) wp_die();
        AutoSEO_DB::clear_logs();
        wp_send_json_success();
    }

    public function process( $post_id, $post_title, $post_content, $post_type = 'post', $action = 'manual' ) {
        $overwrite = get_option( 'autoseo_overwrite', '0' ) === '1';
        $result    = $this->generator->generate( $post_id, $post_title, $post_content, $post_type );

        if ( is_wp_error( $result ) ) {
            AutoSEO_DB::log( $post_id, $post_title, $action, array(), 'error', $result->get_error_message() );
            return $result;
        }

        $this->save_meta( $post_id, '_yoast_wpseo_title',    $result['seo_title'],       $overwrite );
        $this->save_meta( $post_id, '_yoast_wpseo_metadesc', $result['seo_description'], $overwrite );
        $this->save_meta( $post_id, '_yoast_wpseo_focuskw',  $result['focus_keyphrase'], $overwrite );

        AutoSEO_DB::log( $post_id, $post_title, $action, $result, 'success' );
        return $result;
    }

    private function save_meta( $post_id, $key, $value, $overwrite ) {
        if ( empty( $value ) ) return;
        $existing = get_post_meta( $post_id, $key, true );
        if ( empty( $existing ) || $overwrite ) {
            update_post_meta( $post_id, $key, $value );
        }
    }

    private function has_all_fields( $post_id ) {
        return ! empty( get_post_meta( $post_id, '_yoast_wpseo_title', true ) )
            && ! empty( get_post_meta( $post_id, '_yoast_wpseo_metadesc', true ) )
            && ! empty( get_post_meta( $post_id, '_yoast_wpseo_focuskw', true ) );
    }

    private function missing_fields( $post_id ) {
        $m = array();
        if ( empty( get_post_meta( $post_id, '_yoast_wpseo_title', true ) ) )    $m[] = 'SEO-Titel';
        if ( empty( get_post_meta( $post_id, '_yoast_wpseo_metadesc', true ) ) ) $m[] = 'Meta-Beschreibung';
        if ( empty( get_post_meta( $post_id, '_yoast_wpseo_focuskw', true ) ) )  $m[] = 'Fokus-Keyphrase';
        return $m;
    }
}
