<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class AutoSEO_Generator {

    private $language;

    // Stopwörter Deutsch
    private $stopwords_de = array(
        'der','die','das','den','dem','des','ein','eine','einen','einer','eines',
        'und','oder','aber','für','mit','von','zu','auf','in','ist','sind','wird',
        'werden','hat','haben','wie','was','wo','wer','sich','auch','bei','nach',
        'aus','an','so','als','nicht','noch','alle','mehr','über','unter','im',
        'am','zum','zur','bis','durch','beim','vom','ins','ans','ums','ab','ob'
    );

    // Stopwörter Englisch
    private $stopwords_en = array(
        'the','a','an','and','or','but','for','with','from','to','on','in','is',
        'are','was','were','has','have','how','what','where','who','also','at',
        'by','all','more','this','that','not','its','it','be','do','did','does',
        'will','would','could','should','may','might','can','shall','about','into'
    );

    public function __construct() {
        $this->language = get_option( 'autoseo_language', 'de' );
    }

    /**
     * Hauptfunktion: Alle SEO-Felder generieren
     */
    public function generate( $post_id, $post_title, $post_content, $post_type = 'post' ) {
        $categories = $this->get_terms( $post_id, $post_type );
        $keywords   = $this->extract_keywords( $post_title, $post_content );
        $site_name  = get_bloginfo( 'name' );

        return array(
            'seo_title'       => $this->build_title( $post_title, $site_name ),
            'seo_description' => $this->build_description( $post_title, $post_content, $site_name ),
            'focus_keyphrase' => $this->build_keyphrase( $post_title, $keywords, $categories ),
        );
    }

    /**
     * SEO-Titel: max. 60 Zeichen
     * Format: "Beitragstitel | Seitenname" oder nur Titel wenn zu lang
     */
    private function build_title( $post_title, $site_name ) {
        // Sonderzeichen bereinigen
        $title = trim( $post_title );

        $with_site = $title . ' | ' . $site_name;
        if ( mb_strlen( $with_site ) <= 60 ) {
            return $with_site;
        }

        // Titel kürzen damit Sitename passt
        $max_title = 60 - mb_strlen( ' | ' . $site_name );
        if ( $max_title > 20 ) {
            return mb_substr( $title, 0, $max_title - 3 ) . '... | ' . $site_name;
        }

        // Nur Titel ohne Sitename
        if ( mb_strlen( $title ) <= 60 ) {
            return $title;
        }
        return mb_substr( $title, 0, 57 ) . '...';
    }

    /**
     * Meta-Beschreibung: max. 155 Zeichen
     * Extrahiert ersten sinnvollen Satz aus dem Content
     */
    private function build_description( $post_title, $post_content, $site_name ) {
        $clean = $this->clean_content( $post_content );

        // Ersten vollständigen Satz finden (mind. 40, max. 130 Zeichen)
        if ( ! empty( $clean ) ) {
            preg_match( '/^(.{40,130}?[.!?])(\s|$)/u', $clean, $matches );
            if ( ! empty( $matches[1] ) ) {
                $sentence = trim( $matches[1] );
                // Sitename anhängen wenn Platz
                $with_site = $sentence . ' » ' . $site_name;
                if ( mb_strlen( $with_site ) <= 155 ) return $with_site;
                if ( mb_strlen( $sentence ) <= 155 )  return $sentence;
                return mb_substr( $sentence, 0, 152 ) . '...';
            }

            // Kein Satzende – ersten Abschnitt nehmen
            $excerpt = mb_substr( $clean, 0, 120 );
            $with_site = $excerpt . ' » ' . $site_name;
            if ( mb_strlen( $with_site ) <= 155 ) return $with_site;
            return mb_substr( $excerpt, 0, 152 ) . '...';
        }

        // Fallback: Aus Titel
        $cta = $this->get_cta( $site_name );
        $desc = $post_title . ' – ' . $cta;
        if ( mb_strlen( $desc ) <= 155 ) return $desc;
        return mb_substr( $desc, 0, 152 ) . '...';
    }

    /**
     * Fokus-Keyphrase: 2-3 Wörter aus dem Titel
     */
    private function build_keyphrase( $post_title, $keywords, $categories ) {
        $stopwords = ( $this->language === 'de' ) ? $this->stopwords_de : $this->stopwords_en;

        // Titel-Wörter filtern
        $words = preg_split( '/[\s\-–—\/]+/u', mb_strtolower( $post_title ) );
        $filtered = array();
        foreach ( $words as $w ) {
            $w = preg_replace( '/[^a-zäöüßa-z0-9]/ui', '', $w );
            if ( mb_strlen( $w ) >= 4 && ! in_array( $w, $stopwords ) ) {
                $filtered[] = $w;
            }
        }

        // 2-3 Wörter nehmen
        $phrase = array_slice( $filtered, 0, 3 );

        // Kategorie voranstellen wenn nur 1-2 Wörter
        if ( count( $phrase ) < 2 && ! empty( $categories ) ) {
            $cat = mb_strtolower( $categories[0] );
            if ( ! in_array( $cat, $phrase ) && mb_strlen( $cat ) >= 3 ) {
                array_unshift( $phrase, $cat );
            }
        }

        if ( empty( $phrase ) ) {
            return mb_strtolower( mb_substr( $post_title, 0, 25 ) );
        }

        return implode( ' ', array_slice( $phrase, 0, 3 ) );
    }

    /**
     * Häufigste Keywords aus Text extrahieren
     */
    private function extract_keywords( $post_title, $post_content ) {
        $stopwords = ( $this->language === 'de' ) ? $this->stopwords_de : $this->stopwords_en;
        $text  = mb_strtolower( $post_title . ' ' . $this->clean_content( $post_content ) );
        $text  = preg_replace( '/[^a-zäöüßa-z\s]/ui', ' ', $text );
        $words = preg_split( '/\s+/', $text );
        $freq  = array();
        foreach ( $words as $w ) {
            if ( mb_strlen( $w ) < 4 || in_array( $w, $stopwords ) ) continue;
            $freq[ $w ] = ( $freq[ $w ] ?? 0 ) + 1;
        }
        arsort( $freq );
        return array_keys( array_slice( $freq, 0, 10 ) );
    }

    /**
     * Content bereinigen
     */
    private function clean_content( $content ) {
        $text = wp_strip_all_tags( $content );
        $text = preg_replace( '/\[.*?\]/s', '', $text );   // Shortcodes
        $text = preg_replace( '/https?:\/\/\S+/', '', $text ); // URLs
        $text = preg_replace( '/\s+/', ' ', $text );
        return trim( $text );
    }

    /**
     * Kategorien/Tags holen
     */
    private function get_terms( $post_id, $post_type ) {
        $result = array();
        if ( $post_type === 'post' ) {
            foreach ( get_the_category( $post_id ) as $cat ) {
                $result[] = $cat->name;
            }
        }
        foreach ( get_object_taxonomies( $post_type ) as $tax ) {
            $terms = get_the_terms( $post_id, $tax );
            if ( $terms && ! is_wp_error( $terms ) ) {
                foreach ( $terms as $t ) $result[] = $t->name;
            }
        }
        return array_unique( $result );
    }

    /**
     * Call-to-Action je nach Sprache
     */
    private function get_cta( $site_name ) {
        $ctas_de = array(
            'Jetzt auf ' . $site_name . ' entdecken!',
            'Alle Infos auf ' . $site_name . '.',
            'Mehr dazu auf ' . $site_name . '.',
            'Kostenlos auf ' . $site_name . ' lesen.',
        );
        $ctas_en = array(
            'Discover it now on ' . $site_name . '!',
            'Read more on ' . $site_name . '.',
            'Find out more at ' . $site_name . '.',
        );
        $ctas = ( $this->language === 'de' ) ? $ctas_de : $ctas_en;
        return $ctas[ array_rand( $ctas ) ];
    }
}
